"""
This script is used to measure the backlash of the Scan Table.
"""

import config_enum.scan_table_enum as scan_table_enum

from predefined_tasks.common.helper import send_to_gc
from predefined_tasks.common.helper import initialize_instrument

from py_pli.pylib import VUnits
from py_pli.pylib import Measurements
from py_pli.pylib import GlobalVar

from virtualunits.HAL import HAL
from meas_services.abs import ABSFService

from pylog.pylogger import PyLogger

hal_unit: HAL = VUnits.instance.hal
meas_unit = hal_unit.measurementUnit
plate_door = hal_unit.plateDoor
scan_table = hal_unit.scan_table
absf_meas: ABSFService = Measurements.instance.absfMeas


async def scan_table_backlash(scan_radius=4.5, scan_step=0.001):
    """
    Measure the backlash of the Scan Table.
    Script-Preconditions:
        96 Test Plate is loaded
        Filter IAT (ID "1006") is loaded.

    Args:
        scan_radius (float): The scan radius around the well center (default 4.5 mm).
        scan_step (float): The scan step distance (default 0.001 mm).

    Returns:
        tuple: Backlash in X, Y.
    """
    GlobalVar.set_stop_gc(False)

    await send_to_gc(f"Initializing Instrument")
    await initialize_instrument()
    await absf_meas.test_init(filter_module_id="1006")

    scan_table.ReloadConfig()

    # Set default parameter values
    scan_radius = float(scan_radius) if (scan_radius != '') else 4.5
    scan_step = float(scan_step) if (scan_step != '') else 0.001

    well_x = 1
    well_y = 1

    await send_to_gc(f"Measuring Scan Table Backlash")

    scan_table.SetPlateType("'96 Test Plate'")
    scan_table.SetCurrentMeasPosition(scan_table_enum.GC_Params.FBDTop_TopLeftCorner)
    await absf_meas.preExecute()

    #TODO Test Slow or Homing Profile for higher precision
    # await scan_table.use_profile_operation()
    # await scan_table.use_profile_slow()
    await scan_table.use_profile_reset()
    
    offset_range = [offset / 1e6 for offset in range(round(-scan_radius * 1e6), round((scan_radius + scan_step) * 1e6), round(scan_step * 1e6))]

    x_scan_positive = {}

    for x_offset in offset_range:
        if GlobalVar.get_stop_gc():
            await send_to_gc(f"Script stopped by user", log=True, error=True)
            return f"gc_predefined_tasks> Execution stopped."

        await scan_table.MoveToWell(well_x, well_y, x_offset, 0.0)
        well_data = await absf_meas.execute()
        x_scan_positive[x_offset] = well_data.signal
        
    #TODO Test with average center position
    # x_center_positive = find_center(x_scan_positive)
    x_center_positive = find_mean_center(x_scan_positive)

    x_scan_negative = {}

    for x_offset in offset_range:
        if GlobalVar.get_stop_gc():
            await send_to_gc(f"Script stopped by user", log=True, error=True)
            return f"gc_predefined_tasks> Execution stopped."
            
        await scan_table.MoveToWell(well_x, well_y, -x_offset, 0.0)
        well_data = await absf_meas.execute()
        x_scan_negative[-x_offset] = well_data.signal
        
    x_center_negative = find_center(x_scan_negative)
    
    y_scan_positive = {}

    for y_offset in offset_range:
        if GlobalVar.get_stop_gc():
            await send_to_gc(f"Script stopped by user", log=True, error=True)
            return f"gc_predefined_tasks> Execution stopped."

        await scan_table.MoveToWell(well_x, well_y, 0.0, y_offset)
        well_data = await absf_meas.execute()
        y_scan_positive[y_offset] = well_data.signal
        
    y_center_positive = find_center(y_scan_positive)
    
    y_scan_negative = {}

    for y_offset in offset_range:
        if GlobalVar.get_stop_gc():
            await send_to_gc(f"Script stopped by user", log=True, error=True)
            return f"gc_predefined_tasks> Execution stopped."
            
        await scan_table.MoveToWell(well_x, well_y, 0.0, -y_offset)
        well_data = await absf_meas.execute()
        y_scan_negative[-y_offset] = well_data.signal
        
    y_center_negative = find_center(y_scan_negative)

    backlash_x = x_center_positive - x_center_negative
    backlash_y = y_center_positive - y_center_negative

    await scan_table.Home()

    await send_to_gc(f"offset ; x_scan_positive ; x_scan_negative ; y_scan_positive ; y_scan_negative")
    for offset in offset_range:
        await send_to_gc(f"{offset:6.3f} ; {x_scan_positive[offset]:15.0f} ; {x_scan_negative[offset]:15.0f} ; {y_scan_positive[offset]:15.0f} ; {y_scan_negative[offset]:15.0f}")

    await send_to_gc(f"")
    await send_to_gc(f"x_center_positive: {x_center_positive:6.3f}")
    await send_to_gc(f"x_center_negative: {x_center_negative:6.3f}")
    await send_to_gc(f"y_center_positive: {y_center_positive:6.3f}")
    await send_to_gc(f"y_center_negative: {y_center_negative:6.3f}")
    await send_to_gc(f"")
    await send_to_gc(f"Scan Table Backlash = {backlash_x:.3f} ; {backlash_y:.3f}", log=True)

    return f"gc_predefined_tasks> Execution completed."


def find_center(scan_data: dict):
    limit = max(scan_data.values()) * 0.6
    offsets = [offset for offset, signal in scan_data.items() if signal > limit]
    center = (max(offsets) + min(offsets)) / 2

    return center


def find_mean_center(scan_data: dict):
    center = []
    maximum = max(scan_data.values())
    limit_range = [maximum * percent / 100 for percent in range(20, 81, 10)]
    for limit in limit_range:
        offsets = [offset for offset, signal in scan_data.items() if signal > limit]
        center.append((max(offsets) + min(offsets)) / 2)

    PyLogger.logger.info(f"find_center(): {center}")

    return sum(center) / len(center)

